import time
import LPS_module

FALSE = 0
TRUE = not FALSE
TESTVERSION = "1.12 2025-09-29"

activeDevices = list(range(64))
nActive = 0
STATUS_OK = 0
BAD_PARAMETER = 0x80010000          # out of range input -- frequency outside min/max etc.
BAD_HID_IO = 0x80020000             # an error was returned by the lower level USB drivers
DEVICE_NOT_READY = 0x80030000       # device isn't open, no handle, etc.
FEATURE_NOT_SUPPORTED = 0x80040000  # the selected Lab Brick does not support this function
                                    # Profiles and Bi-directional ramps are only supported in
                                    # newer LPS models
ERROR_STATUS_MSK = 0x80000000       # the MSB is set for errors

# The C library we're calling will be expecting us to have a list
# if active devices. In C this is an array of zeroes but in Python we'll
# substitute a list. The C header defines 'MAXDEVICES' which for the 1.02
# library was 64. So we'll make our list that long.
#
# One way of doing this is the following three lines:
#activeDevices = list(range(64))
#for i in range (0, len(activeDevices)):
#    activeDevices[i] = 0
# But here's a more fun way. :)
activeDevices = [0] * 64

# ---------------- Helper programs ----------------
#
# convert from our integer 10Hz representation to a floating point value in GHz
def FreqInGHz(frequency10Hz):
    return frequency10Hz / 1.0e8

# Takes a number and masks off all but the lowest 32 bits
def Low32(n):
    return n & 0xFFFFFFFF

def AnyError(returnedResult):
    # Because of the 32-bit nature of the C code, we have conversion
    # issues getting values back and forth to/from C. Error values have the
    # high order bit set and are defined up above. We'll mask off all except
    # the lower 32 bits.
    #
    # Returns:
    #   TRUE if the result represents any of the known errors or
    #   FALSE if the result is a valid return value
    temp = Low32(returnedResult)
    if temp == BAD_PARAMETER or temp == BAD_HID_IO or temp == DEVICE_NOT_READY:
        return TRUE
    else:
        return FALSE
    
def SomeSpecificError(returnedResult, error):
    # Because of the 32-bit nature of the C code, we have conversion
    # issues getting values back and forth to/from C. Error values have the
    # high order bit set and are defined up above. We'll mask off all except
    # the lower 32 bits.
    #
    # Returns:
    #   TRUE if the result is the error passed as an argument or
    #   FALSE if the result is a valid return value
    temp = Low32(returnedResult)
    if temp == error:
        return TRUE
    else:
        return FALSE

# Because of the same 32/64 bit madness, the C perror() function gets confused with high bits as well.
# To get around this you have to call the extension like "LPS_module.fnLPS_perror(Low32(result))" but that's tedious
# so this helper does that for you.
def DecodeError(error):
    return LPS_module.fnLPS_perror(Low32(error)).decode("utf-8")

# Humans like to see big numbers in groups of 3 with commas
# This only handles integers!!!
def PrettyPrint(n):
    ntemp = str(abs(n))
    retval = ""
    while len(ntemp) > 0:
        retval = "," + ntemp[-3:] + retval
        ntemp = ntemp[0:-3]
    # We're good to go except for a leading comma.
    retval = retval[1:]
    # If it was negative, fix that now
    if n < 0:
        retval = "-"+retval
    return retval

# ---------------- End of helper programs ----------------

LPS_module.fnLPS_Init()
LPS_module.fnLPS_SetTestMode(FALSE)
nDevices = LPS_module.fnLPS_GetNumDevices()
version = LPS_module.fnLPS_GetLibVersion()
major = (version >> 8) & 0xFF
minor = version & 0xFF
print(f"LPS test/demonstration program version {TESTVERSION}, "
      f"using library version {major}.{minor}, "
      f"and wrapper version {LPS_module.WrapperVersion()}")

print(f"I think there are {nDevices} devices")
if 0 == nDevices:
    c = input("No Vaunix LPS devices located. Would you like to run in test mode? ")
    if c == 'Y' or c == 'y':
        LPS_module.fnLPS_Init()
        LPS_module.fnLPS_SetTestMode(TRUE)
        nDevices = LPS_module.fnLPS_GetNumDevices()
        print()
    elif c == 'N' or c == 'n':
        print("Exiting...")
        exit()

print(f"Found {nDevices} device(s):")

for i in range(1,1+nDevices):
    modelName = LPS_module.fnLPS_GetModelName(i).decode("utf-8")
    print(f"\tDevice {i} is {modelName}")

nActive = LPS_module.fnLPS_GetDevInfo(activeDevices)
print(f"We have {nActive} active devices")

for i in range(0, nActive):
    print(f"\n--------- Device {activeDevices[i]} -------------------------")
    result = LPS_module.fnLPS_InitDevice(activeDevices[i])
    print(f"Opened device {activeDevices[i]}. Returned status: {hex(Low32(result))} ({DecodeError(result)})")

    # --- show the state of the device ---
    modelName = LPS_module.fnLPS_GetModelName(activeDevices[i]).decode("utf-8")
    result = LPS_module.fnLPS_GetSerialNumber(activeDevices[i])
    print(f"Device {activeDevices[i]} ({modelName}) has serial number {result}")

    result = LPS_module.fnLPS_GetNumChannels(activeDevices[i])
    print(f"Max Channels: {result}")

    result = LPS_module.fnLPS_GetDeviceStatus(activeDevices[i])
    print(f"Device Status: {bin(result)}")

    numChannels = LPS_module.fnLPS_GetNumChannels(activeDevices[i])
    for ch in range(numChannels):
        ch += 1
        print(f"--------- Channel {ch}/{numChannels} ------------")

        LPS_module.fnLPS_SetChannel(activeDevices[i], ch)

        result = LPS_module.fnLPS_GetPhaseAngle(activeDevices[i])
        print(f"Current Phase Angle: {result}")

        result = LPS_module.fnLPS_SetPhaseAngle(activeDevices[i], 0)
        if AnyError(result):
            print("  ERROR Setting Phase Angle")
        else:
            print("  Success Setting Phase Angle")

        result = LPS_module.fnLPS_GetPhaseAngle(activeDevices[i])
        print(f"Current Phase Angle: {result}")

        result = LPS_module.fnLPS_SetPhaseAngle(activeDevices[i], 20)
        if AnyError(result):
            print("  ERROR Setting Phase Angle")
        else:
            print("  Success Setting Phase Angle")

        result = LPS_module.fnLPS_GetPhaseAngle(activeDevices[i])
        print(f"Current Phase Angle: {result}")

        result = LPS_module.fnLPS_SetPhaseAngle(activeDevices[i], 40)
        if AnyError(result):
            print("  ERROR Setting Phase Angle")
        else:
            print("  Success Setting Phase Angle")

        result = LPS_module.fnLPS_GetPhaseAngle(activeDevices[i])
        print(f"Current Phase Angle: {result}")

        result = LPS_module.fnLPS_GetWorkingFrequency(activeDevices[i])
        print(f"Current Working Frequency: {result}")

        result = LPS_module.fnLPS_SetWorkingFrequency(activeDevices[i], (LPS_module.fnLPS_GetMinWorkingFrequency(activeDevices[i]) + LPS_module.fnLPS_GetMaxWorkingFrequency(activeDevices[i]) / 2))
        if AnyError(result):
            print(f" ERROR: Status={hex(Low32(result))} ({DecodeError(result)}) in LPS_module.fnLPS_SetWorkingFrequency")
        else:
            print(f"  Set Frequency to {(LPS_module.fnLPS_GetMinWorkingFrequency(activeDevices[i]) + LPS_module.fnLPS_GetMaxWorkingFrequency(activeDevices[i]) / 2)}")

        result = LPS_module.fnLPS_GetWorkingFrequency(activeDevices[i])
        print(f"Current Working Frequency: {result}")

    result = LPS_module.fnLPS_CloseDevice(activeDevices[i])
    print(f"Closed device {activeDevices[i]}. Return Status={hex(Low32(result))} ({DecodeError(result)})")

print("\nEnd of test")
